#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_BloodAvailability : BaseTest 
	{
		private BOL.BloodAvailability _testBloodAv;
		private System.DateTime _currentDate = DateTime.Now;
		private System.Guid _patientGuid;
		

		[SetUp]
		protected void SetUp() 
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
				this._patientGuid = DataCreator.GetRecordGuid("SELECT TOP 1 PatientGuid FROM Patient");

				this.RefreshData = false;
			}

			this._testBloodAv = new BOL.BloodAvailability();
		}


		[Test]
		public void Ctor_Blank_Pass()
		{
			Assert.IsNotNull(this._testBloodAv);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void Ctor_Blank_Fail()
		{
		}

		[Test]
		public void ReportCriteria_Pass()
		{
			this._testBloodAv.AddABORhFilter("A Pos");
			this._testBloodAv.AddABORhFilter("A Neg");
			this._testBloodAv.AddComponentClassFilter("1");
			this._testBloodAv.AddComponentClassFilter("2");
			this._testBloodAv.AddUnitAntigenFilter("6");
			this._testBloodAv.AddUnitAntigenFilter("7");
			this._testBloodAv.AddUnitStatusFilter("A");
			this._testBloodAv.AddUnitStatusFilter("L");
			this._testBloodAv.AddProductAttributeFilter("V");
			this._testBloodAv.AddProductAttributeFilter("W");
			this._testBloodAv.AddBiohazardousFilter(true);
			this._testBloodAv.AddCMVTestingFilter(true);
			this._testBloodAv.AddSickleCellTestingFilter(true);
			this._testBloodAv.AddQuarantineFilter(true);
			this._testBloodAv.AddDonationTypeFilter("A");
			this._testBloodAv.AddDonationTypeFilter("D");
			this._testBloodAv.AddDonationTypeFilter("N");
			//
			foreach( BOL.Division div in new BOL.VbecsUser(Environment.UserName).DivisionsAndRoles.EffectiveDivisions )
			{	
				this._testBloodAv.AddDivisionFilter(div.DivisionCode);
			}
			//
			this._testBloodAv.AddExpirationDateBeforeFilter(DateTime.Today.AddMonths(1));
			this._testBloodAv.AddExpirationDateAfterFilter(DateTime.Today.AddMonths(-1));
			//
			this._testBloodAv.AddPatientFilter(new BOL.Patient(this._patientGuid));
			//
			string reportCriteria = this._testBloodAv.ReportCriteria;
			DataRow drClass = BOL.ReferenceData.GetComponentClass(1);
			DataRow drAntigen = BOL.ReferenceData.GetAntigenType(6);
			DataRow drStatus = BOL.ReferenceData.GetUnitStatus("A");
			DataRow drAttribute = BOL.ReferenceData.GetProductModification("W");
			//
			Assert.IsTrue((reportCriteria.IndexOf("A Pos")>-1),"Report criteria contains ABO/Rh");
			Assert.IsTrue((reportCriteria.IndexOf(drClass[TABLES.ComponentClass.ComponentClassShortName].ToString())>-1), "Report criteria contains component class");
			Assert.IsTrue((reportCriteria.IndexOf(drAntigen[TABLES.AntigenType.AntigenTypeName].ToString())>-1), "Report criteria contains antigen type");
			Assert.IsTrue((reportCriteria.IndexOf(drStatus[TABLES.UnitStatus.UnitStatusText].ToString())>-1), "Report criteria contains unit status");
			Assert.IsTrue((reportCriteria.IndexOf(drAttribute[TABLES.ProductModification.ProductModificationText].ToString())>-1), "Report criteria contains product attribute");
		}

		[Test]
		public void ReportCriteria_Fail()
		{
			//No criteria set.
			string reportCriteria = this._testBloodAv.ReportCriteria;
			//
			DataRow drAvailableStatus = BOL.ReferenceData.GetUnitStatus("A");
			DataRow drLimitedStatus = BOL.ReferenceData.GetUnitStatus("L");
			DataRow drAssignedStatus = BOL.ReferenceData.GetUnitStatus("S");
			DataRow drCrossmatchedStatus = BOL.ReferenceData.GetUnitStatus("C");
			DataRow drIssuedStatus = BOL.ReferenceData.GetUnitStatus("I");
			//
			Assert.IsTrue((reportCriteria.IndexOf(drLimitedStatus[TABLES.UnitStatus.UnitStatusText].ToString())>-1), "Report criteria contains limited status");
			Assert.IsTrue((reportCriteria.IndexOf(drAvailableStatus[TABLES.UnitStatus.UnitStatusText].ToString())>-1), "Report criteria contains available status");
			Assert.IsTrue((reportCriteria.IndexOf(drAssignedStatus[TABLES.UnitStatus.UnitStatusText].ToString())>-1), "Report criteria contains assigned status");
			Assert.IsTrue((reportCriteria.IndexOf(drCrossmatchedStatus[TABLES.UnitStatus.UnitStatusText].ToString())>-1), "Report criteria contains crossmatched status");
			Assert.IsTrue((reportCriteria.IndexOf(drIssuedStatus[TABLES.UnitStatus.UnitStatusText].ToString())>-1), "Report criteria contains issued status");
		}
		
		[Test]
		public void GetBloodAvailabilityCount_DailyInventory_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsDailyInventoryAllReport = true;
			int count = this._testBloodAv.GetBloodAvailabilityCount();
			Assert.IsNotNull(count);
		}
		
		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodAvailabilityCount_DailyInventory_Fail()
		{
		}

		[Test]
		public void GetBloodAvailability_DailyInventory_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsDailyInventoryAllReport = true;
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.IsTrue(report.Tables.Count > 0);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodAvailability_DailyInventory_Fail()
		{
		}

		[Test]
		public void GetBloodAvailabilityCount_CustomReport_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddUnitStatusFilter("A");
			this._testBloodAv.CustomReportFilterStatement = this._testBloodAv.BuildFilterStatement();
			//
			int count = this._testBloodAv.GetBloodAvailabilityCount();
			Assert.IsNotNull(count);
		}

		[Test]
		[ExpectedException(typeof(BusinessObjectException))]
		public void GetBloodAvailabilityCount_CustomReport_Fail()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsCustomReport = true;
			int count = this._testBloodAv.GetBloodAvailabilityCount();
			Assert.IsNotNull(count);
		}

		[Test]
		public void GetBloodAvailability_CustomReport_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddUnitStatusFilter("A");
			this._testBloodAv.CustomReportFilterStatement = this._testBloodAv.BuildFilterStatement();
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.IsTrue(report.Tables.Count > 0);
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void GetBloodAvailability_CustomReport_Fail()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsCustomReport = true;
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.IsTrue(report.Tables.Count > 0);
		}

		[Test]
		public void GetBloodAvailability_CustomReport_Empty()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddExpirationDateAfterFilter(DateTime.Today.AddYears(1000));
			this._testBloodAv.CustomReportFilterStatement = this._testBloodAv.BuildFilterStatement();
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.AreEqual(0, report.Tables[0].Rows.Count);
		}

		[Test]
		public void GetBloodAvailabilityCount_AntigenTypedUnits_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsAntigenTypedInventoryReport = true;
			int count = this._testBloodAv.GetBloodAvailabilityCount();
			Assert.IsNotNull(count);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodAvailabilityCount_AntigenTypedUnits_Fail()
		{
		}

		[Test]
		public void GetBloodAvailability_AntigenTypedUnits_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsAntigenTypedInventoryReport = true;
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.IsTrue(report.Tables.Count > 0);
		}
		
		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodAvailability_AntigenTypedUnits_Fail()
		{
		}

		[Test]
		public void GetBloodAvailability_UnitsWithNoDisposition_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsNoDispositionReport = true;
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.IsTrue(report.Tables.Count > 0);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodAvailability_UnitsWithNoDisposition_Fail()
		{
		}

		[Test]
		public void GetBloodAvailabilityCount_UnitsWithNoDisposition_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsNoDispositionReport = true;
			int count = this._testBloodAv.GetBloodAvailabilityCount();
			Assert.IsNotNull(count);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodAvailabilityCount_UnitsWithNoDisposition_Fail()
		{
		}

		[Test]
		public void GetBloodAvailabilityCount_UnitsWithFinalDisposition_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsFinalDispositionReport = true;
			this._testBloodAv.FinalDispositionStartDate = DateTime.Now.AddDays(-30);
			this._testBloodAv.FinalDispositionEndDate = DateTime.Now;
			int count = this._testBloodAv.GetBloodAvailabilityCount();
			Assert.IsNotNull(count);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodAvailabilityCount_UnitsWithFinalDisposition_Fail()
		{
		}

		[Test]
		public void GetBloodAvailability_UnitsWithFinalDisposition_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsFinalDispositionReport = true;
			this._testBloodAv.FinalDispositionStartDate = DateTime.Now.AddDays(-30);
			this._testBloodAv.FinalDispositionEndDate = DateTime.Now;
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.IsTrue(report.Tables.Count > 0);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodAvailability_UnitsWithFinalDisposition_Fail()
		{
		}

		[Test]
		public void GetBloodAvailability_IndividualPatient_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsIndividualPatientReport = true;
			this._testBloodAv.PatientGuid = this._patientGuid;
			//
			//Ensure that a unit is issued to this patient - for 100% coverage
			this.IssueUnitToPatient(this._patientGuid);
			//
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.IsTrue(report.Tables.Count > 0);
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void GetBloodAvailability_IndividualPatient_Fail()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsIndividualPatientReport = true;
			DataSet report = this._testBloodAv.GetBloodAvailabilityReport();
			Assert.IsTrue(report.Tables.Count > 0);
		}

		[Test]
		public void GetBloodAvailabilityCount_IndividualPatient_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsIndividualPatientReport = true;
			this._testBloodAv.PatientGuid = this._patientGuid;
			int count = this._testBloodAv.GetBloodAvailabilityCount();
			Assert.IsNotNull(count);
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void GetBloodAvailabilityCount_IndividualPatient_Fail()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsIndividualPatientReport = true;
			int count = this._testBloodAv.GetBloodAvailabilityCount();
			Assert.IsNotNull(count);
		}

		[Test]
		public void AddComponentClassFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddComponentClassFilter("1");
			DataRow dr = BOL.ReferenceData.GetComponentClass(1);
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("b.ComponentClassId IN ('1')")>-1),"Filter statement contains component class");
		}

		[Test]
		public void AddComponentClassFilter_Fail()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddComponentClassFilter("1");
			string filterBefore = this._testBloodAv.BuildFilterStatement();
			this._testBloodAv.AddComponentClassFilter("1");
			string filterAfter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(filterBefore, filterAfter);
		}

		[Test]
		public void AddABORhFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddABORhFilter("A Pos");
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("b.BloodTypeCode='A' AND b.RhFactorCode='P'")>-1),"Filter statement contains ABO/Rh");
		}
		
		[Test]
		public void AddABORhFilter_Fail()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddABORhFilter("A Pos");
			string filterBefore = this._testBloodAv.BuildFilterStatement();
			this._testBloodAv.AddABORhFilter("A Pos");
			string filterAfter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(filterBefore, filterAfter);
		}

		[Test]
		public void AddDonationTypeFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddDonationTypeFilter("N");
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("dt.RestrictionTypeCode IN ('N')")>-1),"Filter statement contains Donation Type");
		}
		
		[Test]
		public void AddDonationTypeFilter_Fail()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddDonationTypeFilter("N");
			string filterBefore = this._testBloodAv.BuildFilterStatement();
			this._testBloodAv.AddDonationTypeFilter("N");
			string filterAfter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(filterBefore, filterAfter);
		}

		[Test]
		public void AddUnitStatusFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddUnitStatusFilter("A");
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("b.UnitStatusCode IN ('A')")>-1),"Filter statement contains Unit Status");
		}

		[Test]
		public void AddUnitStatusFilter_Fail()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddUnitStatusFilter("A");
			string filterBefore = this._testBloodAv.BuildFilterStatement();
			this._testBloodAv.AddUnitStatusFilter("A");
			string filterAfter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(filterBefore, filterAfter);
		}

		[Test]
		public void AddUnitAntigenFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddUnitAntigenFilter("6");
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("a.AntigenTypeId IN ('6')")>-1),"Filter statement contains Unit Antigen Type");
		}
		
		[Test]
		public void AddUnitAntigenFilter_Fail()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddUnitAntigenFilter("6");
			string filterBefore = this._testBloodAv.BuildFilterStatement();
			this._testBloodAv.AddUnitAntigenFilter("6");
			string filterAfter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(filterBefore, filterAfter);
		}

		[Test]
		public void AddProductAttributeFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddProductAttributeFilter("W");
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('W')")>-1),"Filter statement contains Product Attribute");
		}
		
		[Test]
		public void AddProductAttributeFilter_Fail()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddProductAttributeFilter("W");
			string filterBefore = this._testBloodAv.BuildFilterStatement();
			this._testBloodAv.AddProductAttributeFilter("W");
			string filterAfter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(filterBefore, filterAfter);
		}

		[Test]
		public void AddDivisionFilter_Pass()
		{
			string tmpDiv = this.TestDivisionCode.Trim();

			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddDivisionFilter(tmpDiv);
			string filter = this._testBloodAv.BuildFilterStatement();

			Assert.IsTrue((filter.IndexOf("b.DivisionCode IN ('" + tmpDiv + "')")>-1),"Filter statement contains Division Code");
		}
		
		[Test]
		public void AddDivisionFilter_Fail()
		{
			string tmpDiv = this.TestDivisionCode.Trim();
			
			this._testBloodAv.IsCustomReport = true;

			this._testBloodAv.AddDivisionFilter(tmpDiv);
			string filterBefore = this._testBloodAv.BuildFilterStatement();
			
			this._testBloodAv.AddDivisionFilter(tmpDiv);
			string filterAfter = this._testBloodAv.BuildFilterStatement();

			Assert.AreEqual(filterBefore, filterAfter);
		}

		[Test]
		public void AddExpirationDateBeforeFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddExpirationDateBeforeFilter(_currentDate);
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("b.UnitExpirationDate) < '"+_currentDate.Date.AddHours(23).AddMinutes(59).AddSeconds(59).ToString()+"'")>-1),"Filter statement contains Expiration Before Date");
		}
		
		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void AddExpirationDateBeforeFilter_Fail()
		{
		}

		[Test]
		public void AddExpirationDateAfterFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddExpirationDateAfterFilter(_currentDate);
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("b.UnitExpirationDate) > '"+_currentDate.Date.ToString()+"'")>-1),"Filter statement contains Expiration After Date");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void AddExpirationDateAfterFilter_Fail()
		{
		}

		[Test]
		public void AddCMVTestingFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddCMVTestingFilter(true);
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("(t.BloodTestTypeId=103 AND t.TestResultId='N' AND t.RecordStatusCode = 'A')")>-1),"Filter statement contains CMV Neg");
			//
			this._testBloodAv.AddCMVTestingFilter(false);
			filter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(-1,filter.IndexOf("(t.BloodTestTypeId=103 AND t.TestResultId='P' AND t.RecordStatusCode = 'A')"));
		}
		
		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void AddCMVTestingFilter_Fail()
		{
		}

		[Test]
		public void AddSickleCellTestingFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddSickleCellTestingFilter(true);
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("(t.BloodTestTypeId=104 AND t.TestResultId='N' AND t.RecordStatusCode = 'A')")>-1),"Filter statement contains Sickle Cell Neg");
			//
			this._testBloodAv.AddSickleCellTestingFilter(false);
			filter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(-1,filter.IndexOf("(t.BloodTestTypeId=104 AND t.TestResultId='P' AND t.RecordStatusCode = 'A')"));
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void AddSickleCellTestingFilter_Fail()
		{
		}

		[Test]
		public void AddQuarantineFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddQuarantineFilter(true);
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("b.QuarantineIndicator=1")>-1),"Filter statement contains Quarantine");
			//
			this._testBloodAv.AddQuarantineFilter(false);
			filter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(-1,filter.IndexOf("b.QuarantineIndicator=0"));

		}
		
		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void AddQuarantineFilter_Fail()
		{
		}

		[Test]
		public void AddBiohazardousFilter_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddBiohazardousFilter(true);
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("b.BiohazardousWarningIndicator=1")>-1),"Filter statement contains Biohazardous");
			//
			this._testBloodAv.AddBiohazardousFilter(false);
			filter = this._testBloodAv.BuildFilterStatement();
			Assert.AreEqual(-1,filter.IndexOf("b.BiohazardousWarningIndicator=0"));

		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void AddBiohazardousFilter_Fail()
		{
		}

		[Test]
		public void AddPatientFilter_Pass()
		{
			this._testBloodAv.IsIndividualPatientReport = true;
			this._testBloodAv.AddPatientFilter(new BOL.Patient(_patientGuid));
			string filter = this._testBloodAv.BuildFilterStatement();
			Assert.IsTrue((filter.IndexOf("pat.PatientGuid='"+_patientGuid.ToString()+"'")>-1),"Filter statement contains Patient");
		}

		[Test]
		public void AddPatientFilter_Fail()
		{
			try
			{
				this._testBloodAv.AddPatientFilter(null);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
			try
			{
				this._testBloodAv.AddPatientFilter(new BOL.Patient(Guid.Empty));
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		[Test]
		public void BuildFilterStatement_Custom_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			//
			//Adding all blood types for 100% coverage!
			this._testBloodAv.AddABORhFilter("A");
			this._testBloodAv.AddABORhFilter("A Pos");
			this._testBloodAv.AddABORhFilter("A Neg");
			this._testBloodAv.AddABORhFilter("B");
			this._testBloodAv.AddABORhFilter("B Pos");
			this._testBloodAv.AddABORhFilter("B Neg");
			this._testBloodAv.AddABORhFilter("AB");
			this._testBloodAv.AddABORhFilter("AB Pos");
			this._testBloodAv.AddABORhFilter("AB Neg");
			this._testBloodAv.AddABORhFilter("O");
			this._testBloodAv.AddABORhFilter("O Pos");
			this._testBloodAv.AddABORhFilter("O Neg");
			this._testBloodAv.AddABORhFilter("Pooled Pos");
			this._testBloodAv.AddABORhFilter("Pooled Neg");
			this._testBloodAv.AddABORhFilter("Pooled Pooled");
			this._testBloodAv.AddABORhFilter("A Pooled");
			this._testBloodAv.AddABORhFilter("B Pooled");
			this._testBloodAv.AddABORhFilter("AB Pooled");
			this._testBloodAv.AddABORhFilter("O Pooled");
			//
			this._testBloodAv.AddComponentClassFilter("1");
			this._testBloodAv.AddUnitAntigenFilter("6");
			this._testBloodAv.AddUnitStatusFilter("A");
			this._testBloodAv.AddProductAttributeFilter("W");

			string filter = this._testBloodAv.BuildFilterStatement();

			Assert.IsTrue((filter.IndexOf("b.BloodTypeCode='A' AND b.RhFactorCode='P'")>-1),"Filter statement contains ABO/Rh");
			Assert.IsTrue((filter.IndexOf("b.ComponentClassId IN ('1')")>-1),"Filter statement contains component class");
			Assert.IsTrue((filter.IndexOf("a.AntigenTypeId IN ('6')")>-1),"Filter statement contains antigen type");
			Assert.IsTrue((filter.IndexOf("b.UnitStatusCode IN ('A')")>-1),"Filter statement contains unit status");
			Assert.IsTrue((filter.IndexOf("SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('W')")>-1),"Filter statement contains product attribute");
		}

		[Test]
		public void BuildFilterStatement_AntigenTypedReport_Pass()
		{
			this._testBloodAv.IsAntigenTypedInventoryReport = true;
			this._testBloodAv.IsMultiDivisional = true;
			//
			string filter = this._testBloodAv.BuildFilterStatement();
			//
			Assert.IsTrue((filter.IndexOf("b.UnitStatusCode IN ('A','L')")>-1),"Filter statement contains unit status");
		}

		[Test]
		public void BuildFilterStatement_DailyInventoryAvailableReport_Pass()
		{
			this._testBloodAv.IsDailyInventoryAvailableReport = true;
			//
			string filter = this._testBloodAv.BuildFilterStatement();
			//
			Assert.IsTrue((filter.IndexOf("b.UnitStatusCode IN ('L','A')")>-1),"Filter statement contains unit status");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void BuildFilterStatement_Fail()
		{
		}

		[Test]
		public void IsIndividualPatientReport_Pass()
		{
			this._testBloodAv.IsIndividualPatientReport = true;

			Assert.IsTrue(this._testBloodAv.IsIndividualPatientReport);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsIndividualPatientReport_Fail()
		{
		}

		[Test]
		public void IsDailyInventoryAllReport_Pass()
		{
			this._testBloodAv.IsDailyInventoryAllReport = true;

			Assert.IsTrue(this._testBloodAv.IsDailyInventoryAllReport);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsDailyInventoryAllReport_Fail()
		{
		}

		[Test]
		public void IsDailyInventoryAvailableReport_Pass()
		{
			this._testBloodAv.IsDailyInventoryAvailableReport = true;

			Assert.IsTrue(this._testBloodAv.IsDailyInventoryAvailableReport);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsDailyInventoryAvailableReport_Fail()
		{
		}

		[Test]
		public void IsAntigenTypedInventoryReport_Pass()
		{
			this._testBloodAv.IsAntigenTypedInventoryReport = true;

			Assert.IsTrue(this._testBloodAv.IsAntigenTypedInventoryReport);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsAntigenTypedInventoryReport_Fail()
		{
		}

		[Test]
		public void IsNoDispositionReport_Pass()
		{
			this._testBloodAv.IsNoDispositionReport = true;

			Assert.IsTrue(this._testBloodAv.IsNoDispositionReport);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsNoDispositionReport_Fail()
		{
		}

		[Test]
		public void IsFinalDispositionReportPass()
		{
			this._testBloodAv.IsFinalDispositionReport = true;

			Assert.IsTrue(this._testBloodAv.IsFinalDispositionReport);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsFinalDispositionReport_Fail()
		{
		}

		[Test]
		public void FinalDispositionStartDatePass()
		{
			DateTime testDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			this._testBloodAv.FinalDispositionStartDate = testDate;

			Assert.IsTrue(this._testBloodAv.FinalDispositionStartDate == testDate);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void FinalDispositionStartDate_Fail()
		{
		}

		[Test]
		public void FinalDispositionEndDatePass()
		{
			DateTime testDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			this._testBloodAv.FinalDispositionEndDate = testDate;

			Assert.IsTrue(this._testBloodAv.FinalDispositionEndDate == testDate);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void FinalDispositionEndDate_Fail()
		{
		}

		[Test]
		public void PrintDate_Pass()
		{
			DateTime testDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			this._testBloodAv.IsScheduled = true;
			this._testBloodAv.PrintDate = testDate;

			Assert.IsTrue(this._testBloodAv.PrintDate == testDate);
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void PrintDate_Fail()
		{
			DateTime testDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			this._testBloodAv.IsScheduled = false;
			this._testBloodAv.PrintDate = testDate;

			Assert.IsTrue(this._testBloodAv.PrintDate == testDate);
		}

		[Test]
		public void AddFinalDispositionEndDateFilter_Pass()
		{
			DateTime testDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			DateTime newDate = testDate.Date.AddHours(23).AddMinutes(59).AddSeconds(59);

			this._testBloodAv.AddFinalDispositionEndDateFilter(testDate);

			Assert.IsTrue(this._testBloodAv.FinalDispositionEndDate == newDate);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void AddFinalDispositionEndDateFilter_Fail()
		{
		}
		
		[Test]
		public void GenerateReport_DailyInventoryAll_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsDailyInventoryAllReport = true;
            //TODO: Fix once Kman migrates reports.
			//gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport baReport = this._testBloodAv.GenerateReport(Environment.UserName);
            Assert.IsNotNull(this._testBloodAv);
		}

		[Test]
		public void GenerateReport_AntigenTypedUnits_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsAntigenTypedInventoryReport = true;
            //TODO: Fix once Kman finishes migrating reports
			//gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport baReport = this._testBloodAv.GenerateReport(Environment.UserName);
            Assert.IsNotNull(_testBloodAv);
		}

		[Test]
		public void GenerateReport_DailyInventoryAvailable_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsDailyInventoryAvailableReport = true;
            //TODO: Fix once kman finishes migrating reports.
			//gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport baReport = this._testBloodAv.GenerateReport(Environment.UserName);
            Assert.IsNotNull(_testBloodAv);
		}

		[Test]
		public void GenerateReport_FinalDisposition_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsFinalDispositionReport = true;
			this._testBloodAv.FinalDispositionStartDate = DateTime.Now.AddDays(-30);
			this._testBloodAv.FinalDispositionEndDate = DateTime.Now;
            //TODO: Fix once kman finishes migrating reports.
			//gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport baReport = this._testBloodAv.GenerateReport(Environment.UserName);
            Assert.IsNotNull(_testBloodAv);
		}

		[Test]
		public void GenerateReport_IndividualPatient_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsIndividualPatientReport = true;
			this._testBloodAv.PatientGuid = this._patientGuid;
            //TODO: Fix once kman finishes migrating reports.
			//gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport baReport = this._testBloodAv.GenerateReport(Environment.UserName);
            Assert.IsNotNull(_testBloodAv);
		}

		[Test]
		public void GenerateReport_NoDisposition_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsNoDispositionReport = true;
            //TODO: Fix once kman finishes migrating reports.
			//gov.va.med.vbecs.reports.BloodAvailability.BloodAvailabilityReport baReport = this._testBloodAv.GenerateReport(Environment.UserName);
            Assert.IsNotNull(_testBloodAv);
		}

		[Test]
		public void GenerateReport_CustomReport_Pass()
		{
			this._testBloodAv = new BOL.BloodAvailability();
			this._testBloodAv.IsCustomReport = true;
			this._testBloodAv.AddUnitStatusFilter("A");
			this._testBloodAv.CustomReportFilterStatement = this._testBloodAv.BuildFilterStatement();
            Assert.IsNotNull(_testBloodAv);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GenerateReport_Fail()
		{
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GenerateAndPrintReport_Pass()
		{
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GenerateAndPrintReport_Fail()
		{
		}

		[Test]
		public void IsCustomReport_Pass()
		{
			this._testBloodAv.IsCustomReport = true;

			Assert.IsTrue(this._testBloodAv.IsCustomReport);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsCustomReport_Fail()
		{
		}

		[Test]
		public void PatientGuid_Pass()
		{
			this._testBloodAv.IsIndividualPatientReport = true;
			//
			this._testBloodAv.PatientGuid=this._patientGuid;
		
			Assert.AreEqual(this._patientGuid,this._testBloodAv.PatientGuid);
		}

		[Test]
		public void PatientGuid_Fail()
		{
			this._testBloodAv.IsIndividualPatientReport = true;
			//
			this._testBloodAv.SetInitialBrokenRules("DlgPatientAvailabilityType");
			this._testBloodAv.PatientGuid=this._patientGuid;
			int beforeCount = this._testBloodAv.BrokenRulesCount;
			//
			this._testBloodAv.PatientGuid=Guid.Empty;
			int afterCount = this._testBloodAv.BrokenRulesCount;
			//
			Assert.AreEqual(beforeCount+1,afterCount);
		}

		[Test]
		public void CustomReportFilterStatement_Set_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			//
			this._testBloodAv.AddABORhFilter("A Pos");
			this._testBloodAv.AddComponentClassFilter("1");
			this._testBloodAv.AddUnitAntigenFilter("6");
			this._testBloodAv.AddUnitStatusFilter("A");
			this._testBloodAv.AddProductAttributeFilter("W");

			string filter = this._testBloodAv.BuildFilterStatement();

			this._testBloodAv.CustomReportFilterStatement=filter;

			Assert.IsTrue((this._testBloodAv.CustomReportFilterStatement.IndexOf("b.BloodTypeCode='A' AND b.RhFactorCode='P'")>-1),"Filter statement contains ABO/Rh");
			Assert.IsTrue((this._testBloodAv.CustomReportFilterStatement.IndexOf("b.ComponentClassId IN ('1')")>-1),"Filter statement contains component class");
			Assert.IsTrue((this._testBloodAv.CustomReportFilterStatement.IndexOf("a.AntigenTypeId IN ('6')")>-1),"Filter statement contains antigen type");
			Assert.IsTrue((this._testBloodAv.CustomReportFilterStatement.IndexOf("b.UnitStatusCode IN ('A')")>-1),"Filter statement contains unit status");
			Assert.IsTrue((this._testBloodAv.CustomReportFilterStatement.IndexOf("SELECT BloodUnitGuid FROM dbo.fnGetBloodUnitGuidsByProductModificationCode('W')")>-1),"Filter statement contains product attribute");
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void CustomReportFilterStatement_Set_Fail()
		{
			this._testBloodAv.IsCustomReport = false;
			//
			string filter = this._testBloodAv.BuildFilterStatement();
			//
			Assert.AreEqual(filter, this._testBloodAv.CustomReportFilterStatement);
		}

		[Test]
		public void CustomReportFilterStatement_Get_Pass()
		{
			this._testBloodAv.IsCustomReport = true;
			//
			string filter = this._testBloodAv.CustomReportFilterStatement;
			//
			Assert.AreEqual(filter, this._testBloodAv.CustomReportFilterStatement);
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void CustomReportFilterStatement_Get_Fail()
		{
			this._testBloodAv.IsCustomReport = false;
			//
			string filter = this._testBloodAv.CustomReportFilterStatement;
			//
			Assert.AreEqual(filter, this._testBloodAv.CustomReportFilterStatement);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void ClearFilter_Fail()
		{
			this._testBloodAv.IsCustomReport = true;
			//
			string filter = this._testBloodAv.CustomReportFilterStatement;
			//
			Assert.AreEqual(filter, this._testBloodAv.CustomReportFilterStatement);
		}

		[Test]
		public void AddFinalDispositionStartDateFilter_Pass()
		{
			DateTime startDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			//
			this._testBloodAv.AddFinalDispositionStartDateFilter( startDate );
			//
			Assert.AreEqual( this._testBloodAv.FinalDispositionStartDate, startDate.Date );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AddFinalDispositionStartDateFilter_Fail()
		{
		}

		#region Helper Methods

		private void IssueUnitToPatient(Guid patientGuid)
		{
			Guid bloodUnitGuid = Guid.Empty;
			DataTable dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE RecordStatusCode = 'A' AND UnitStatusCode IN ('L','A') AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' ORDER BY LastUpdateDate DESC", true);
			//
			if (dtBloodUnit.Rows.Count > 0)
			{
				bloodUnitGuid = (Guid)dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];
			}
			else
			{
				Assert.AreEqual(bloodUnitGuid, Guid.NewGuid(), "Blood unit does not exist");
				return;
			}
			//
			Guid patientTreatmentGuid = Guid.NewGuid();
			string insertPatientTreatment = "INSERT INTO PATIENTTREATMENT (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Environment.UserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			Guid patientOrderGuid = Guid.NewGuid();
			string insertPatientOrder = "INSERT INTO PATIENTORDER (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Environment.UserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			Guid orderedComponentGuid = Guid.NewGuid();
			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedComponent += "'1',";
            insertOrderedComponent += "'" + DataCreator.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += "'2',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Environment.UserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			Guid orderedUnitGuid = Guid.NewGuid();
			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			Guid bloodUnitTestGuid1 = Guid.NewGuid();
			string insertBloodUnitTest1 = "INSERT INTO BLOODUNITTEST (";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.BloodUnitTestGuid + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.BloodUnitGuid + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.BloodTestTypeId + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.TestDate + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.TestTechId + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.TestResultId + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.EntryTechId + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.EntryMethodCode + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.OrderedComponentGuid + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.OrderedUnitGuid + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.ReactivityPhaseCode + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.RecordStatusCode + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.DivisionCode + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.LastUpdateDate + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.LastUpdateUser + ",";
			insertBloodUnitTest1 += TABLES.BloodUnitTest.LastUpdateFunctionId + ")";
			insertBloodUnitTest1 += "VALUES (";
			insertBloodUnitTest1 += "'" + bloodUnitTestGuid1.ToString() + "',";
			insertBloodUnitTest1 += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitTest1 += "'100',";
			insertBloodUnitTest1 += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest1 += "'" + Environment.UserName + "',";
			insertBloodUnitTest1 += "'0',";
			insertBloodUnitTest1 += "'" + Environment.UserName + "',";
			insertBloodUnitTest1 += "'M',";
			insertBloodUnitTest1 += "'" + orderedComponentGuid.ToString() + "',";
			insertBloodUnitTest1 += "'" + orderedUnitGuid.ToString() + "',";
			insertBloodUnitTest1 += "'2',";
			insertBloodUnitTest1 += "'A',";
			insertBloodUnitTest1 += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertBloodUnitTest1 += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest1 += "'" + Environment.UserName + "',";
			insertBloodUnitTest1 += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			Guid bloodUnitTestGuid2 = Guid.NewGuid();
			string insertBloodUnitTest2 = "INSERT INTO BLOODUNITTEST (";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.BloodUnitTestGuid + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.BloodUnitGuid + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.BloodTestTypeId + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.TestDate + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.TestTechId + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.TestResultId + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.EntryTechId + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.EntryMethodCode + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.OrderedComponentGuid + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.OrderedUnitGuid + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.ReactivityPhaseCode + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.RecordStatusCode + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.DivisionCode + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.LastUpdateDate + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.LastUpdateUser + ",";
			insertBloodUnitTest2 += TABLES.BloodUnitTest.LastUpdateFunctionId + ")";
			insertBloodUnitTest2 += "VALUES (";
			insertBloodUnitTest2 += "'" + bloodUnitTestGuid2.ToString() + "',";
			insertBloodUnitTest2 += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitTest2 += "'102',";
			insertBloodUnitTest2 += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest2 += "'" + Environment.UserName + "',";
			insertBloodUnitTest2 += "'C',";
			insertBloodUnitTest2 += "'" + Environment.UserName + "',";
			insertBloodUnitTest2 += "'M',";
			insertBloodUnitTest2 += "'" + orderedComponentGuid.ToString() + "',";
			insertBloodUnitTest2 += "'" + orderedUnitGuid.ToString() + "',";
			insertBloodUnitTest2 += "'2',";
			insertBloodUnitTest2 += "'A',";
			insertBloodUnitTest2 += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertBloodUnitTest2 += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest2 += "'" + Environment.UserName + "',";
			insertBloodUnitTest2 += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			Guid issuedUnitGuid = Guid.NewGuid();
			string insertIssuedUnit = "INSERT INTO ISSUEDUNIT (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//

			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedComponent);
			UnitTestUtil.RunSQL(insertOrderedUnit);
			UnitTestUtil.RunSQL(insertBloodUnitTest1);
			UnitTestUtil.RunSQL(insertBloodUnitTest2);
			UnitTestUtil.RunSQL(insertIssuedUnit);
		}

		#endregion
	}
}
#endif


